% Kalman filter in discrete time

% notation: the dynamics of the HMM are given by
% X_k = A X_{k-1} + R xi_{k-1}
% Y_k = B X_k + S eta_k
% X_0 is N(X0,Sigma0) distributed

% specify the dimensions of the state spaces of X and Y
p = 4;
q = 2;

% interpretation: 
% the first and second coordinates of X are the position of the particle
% the third and fourth coordinates of X are the velocity of the particle
% the two coordinates of Y are noisy observations of the position of the particle

% specify the dynamics of X
A = [1 0 1 0; 0 1 0 1; 0 0 1 0; 0 0 0 1];
R = 3 * eye(p);

% specify the dynamics of Y
B = [1 0 0 0; 0 1 0 0];
S = 10*eye(q);

% specify the prior mean of X
X0 = [20 20 1 1]';

% specify the prior covariance of X
Sigma0 = 10*eye(p);

% specify the number of time steps
N = 50;

% generate a trajectory of (X,Y)
U = randn(p,N);
V = randn(q,N+1);
X = zeros(p,N+1);
Y = zeros(q,N+1);
X(:,1) = X0 + chol(Sigma0)*randn(p,1);
for k=1:N
   X(:,k+1) = A * X(:,k) + R * U(:,k) ; 
   Y(:,k) = B*X(:,k) + S*V(:,k);
end
Y(:,N+1) = B*X(:,N+1) + S*V(:,N+1);

% run forward filtering recursion 
% notation: \pi_{k+1|k} has normal distribution N(Xpred, Epred)
% notation: \pi_k has normal distribution N(Xhat, Ehat)
Xhat = zeros(p,N+1);
Sigmahat = zeros(p,p,N+1);
% REPLACE THIS BY YOUR CODE
for k = 1:N+1
    if k==1
        Xpred = zeros(p,1);
        Sigmapred = Sigma0;
    else
        Xpred = A*Xhat(:,k-1);
        Sigmapred = A*Sigmahat(:,:,k-1)*A'+R*R';
    end

    K = Sigmapred*B'*inv(B*Sigmapred*B'+S*S');
    Xhat(:,k) = Xpred + K*(Y(:,k) - B*Xpred) ; 
    Sigmahat(:,:,k) = Sigmapred - K*B*Sigmapred;
end
% END REPLACE 

% run backward smoothing recursion
% \pi_{k|N} has normal distribution N(Xsm, Esm)
Xsm = zeros(p,N+1);
Sigmasm = zeros(p,p,N+1);
% REPLACE THIS BY YOUR CODE
Xsm(:,end) = Xhat(:,end);
Sigmasm(:,:,end) = Sigmahat(:,:,end);
for l=1:N
    k = N+1 - l;
    M = inv(A*Sigmahat(:,:,k)*A'+R*R');
    Xsm(:,k) = Xhat(:,k) + Sigmahat(:,:,k)*A'*M*(Xsm(:,k+1)-A*Xhat(:,k));
    Sigmasm(:,:,k) = Sigmahat(:,:,k) - Sigmahat(:,:,k)*A'*M*A*Sigmahat(:,:,k) + Sigmahat(:,:,k)*A'*M*Sigmasm(:,:,k+1)*M*A*Sigmahat(:,:,k);
end
% END REPLACE 

% run forward prediction recursion from n up to N 
% n must be smaller than N
% notation: \pi_{k|n} has normal distribution N(Xpred,Epred)
n = 25;
Xpred = zeros(p,N+1);
Xpred(:,n) = Xhat(:,n);
for k = n+1:N+1
    Xpred(:,k)=A*Xpred(:,k-1);
end

% plot the true particle positions
figure;
xlabel('X1'), ylabel('X2')
plot(X(1,:), X(2,:), 'ks-');
hold on
plot(X(1,1),X(2,1),'ks','MarkerSize',20)
% plot the observed particle positions
plot(Y(1,:), Y(2,:), 'g*');
% plot the filtered particle positions
plot(Xhat(1,:), Xhat(2,:), 'rx:');
% plot the smoothed particle positions
plot(Xsm(1,:), Xsm(2,:), 'bx:');
% plot the predicted particle positions based on the first n observations
plot(Xpred(1,n:end), Xpred(2,n:end), 'cx:');
hold off
legend('true','starting point', 'observed', 'filtered','smoothed','predicted from n = 25','Location','Best')

% save('kalman.mat','X','Y','Xhat','Sigmahat','Xsm','Sigmasm','Xpred')